/* --------------------------------------------------------------
  UploadImageModal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


'use strict';

import Modal from '../libs/Modal'; 
import ImageApi from '../api/ImageApi'; 
import EventsEmitter from '../libs/EventsEmitter';

/**
 * Upload Image Modal Controller
 *
 * This controller will handle the image upload. After a successful upload it will set the newest
 * image to the setting that triggered the upload and then it will also update the select boxes of
 * all the other image widgets, so that the option is also available there.
 */
export default class UploadImageModal {
	/**
	 * Class Constructor 
	 * 
	 * @param {jQuery} $target Select box selector of the setting that triggered the modal.
	 */
	constructor($target) {
		StyleEdit.Validator.isObject($target);
		
		/**
		 * Select Box
		 *
		 * @type {jQuery}
		 */
		this.$target = $target;
		
		/**
		 * Modal Selector
		 *
		 * @type {jQuery}
		 */
		this.$modal = null;
	}
	
	/**
	 * Initialize Controller
	 */
	initialize() {
		const data = {
			title_upload_image_modal: StyleEdit.Language.translate('title_upload_image_modal', 'style_edit'),
			placeholder_browse: StyleEdit.Language.translate('placeholder_browse', 'style_edit'),
			option_upload: StyleEdit.Language.translate('option_upload', 'style_edit'),
			option_cancel: StyleEdit.Language.translate('option_cancel', 'style_edit')
		};
		
		this.$modal = Modal.show($('#upload-image-modal-template'), data);
		
		this._bindEventHandlers();
		
		$.material.init();
		
		EventsEmitter.triggerControllerInitialized(this.$modal, ['UploadImageModal']);
	}
	
	/**
	 * Destroy Controller
	 */
	destroy() {
		Modal.hide(this.$modal);
		EventsEmitter.triggerControllerDestroyed(this.$modal, ['UploadImageModal']);
	}
	
	/**
	 * Event: On Click Upload
	 *
	 * @private
	 */
	_onClickUpload() {
		const $fileInput = this.$modal.find('#image');
		
		$fileInput.parent().removeClass('has-error is-focused');
		
		if ($fileInput.val() === '') {
			$fileInput.parent().addClass('has-error is-focused');
			$fileInput.focus();
			return;
		}
		
		ImageApi.upload($fileInput)
			.done(response => {
				// Append the new file only if it is not already an option.
				if (this.$target.find('option[value="' + response + '"]').length === 0) {
					$('.edit-options .image-list').append(new Option(response, response));
				}
				this.$target.val(response).trigger('change');
				this.destroy();
			})
			.fail((jqxhr, text, error) => {
				Modal.message(StyleEdit.Language.translate('title_unexpected_error', 'style_edit'),
					StyleEdit.Language.translate('message_unexpected_error', 'style_edit'));
				StyleEdit.Debug.log('Style Edit Error: ', jqxhr, text, error);
			});
	}
	
	/**
	 * Event: On Click Cancel
	 *
	 * @private
	 */
	_onClickCancel() {
		this.destroy();
	}
	
	/**
	 * Bind Modal Event Handlers
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$modal
			.on('click', '.btn.upload', () => this._onClickUpload())
			.on('click', '.btn.cancel', () => this._onClickCancel());
	}
}
